#ifndef __SMInterface_h__
#define __SMInterface_h__

#include "SMIplImage.h"
#include "SMManager.h"

#include <string>
#include <map>
#include <boost/interprocess/managed_shared_memory.hpp>

/**	
	This class allows to share timestamped images (IplImage) and data (vector< float>) through shared memory.
	The object creates (or opens) its own shared memory segment. Named data and image queues can be created inside 
	the hared memory and data and images can be pushed into these queus (and read). TSMObject instances can register
	certain queues and also deregister these queues and only the momentarly requested ones can be computed and updated in order to minimize computation.
	
	Each image cueue contains at least single image at all times and older images which are not used by any process are deallocated. 
	
	The data queues have specific length and behave as circular buffers. 
	The queues can contain multiple values with the same timestamp. -- Is this really needed?
	
	Star working: 
	1) TSMObject SM();
	2) registerQueue( const string queue);  
	3) registerQueue( const string queue2);  

	
	To push Image:
	1) IplImage * img = createIplImage( CvSize size, int depth, int channels);
	2) void pushImage( const string queue, img, long long timestamp);
	
	To get newest image:
	1) getNewest( const string queue, long long &timestamp, IplImage *&img);
	2) work with the image
	3) returnImage( img);
	
	To get multiple images with the newest timestamp (the same timestamp):

	1) long long ts = getNewestTimestamp( )
	2) bool success = requestTimestamp( ts);	
	2) void getImage( const string queue, long long timestamp, IplImage *& img);
	3) work with img
	5) void getImage( const string queue, long long timestamp, IplImage *& img2);
	6) work with img and img2
	...
	7) returnTimestamp( long long ts);
	
		
	To push data:
	1) bool worked = push( const std::string, std::vector< float> data, long long timestamp);
	
	
	Get data for timestamp:
	std::vector< std::vector< float> > get( const std::string, long long timestamp);

	Get newest data:
	std::vector< std::vector< float> > getNewest( const std::string, long long &timestamp);

	Get data newer than: 	
	std::multi_map< std::vector< float> > getNewer( const std::string, long long timestamp);
	
	
	The format of the queue identification strings is:
	source_!_operation1_param1_val1_param2_val2_!_operation2_param1_val1 ...
	This format allows sorting according to prefixes and optimization of computation.
	
*/
class TSMObject{

	boost::interprocess::managed_shared_memory smSegment;
	TSMManager *smManager;

	bool destruct;
	std::string memoryName;

	std::map< IplImage *, TSMImage *> tempImages;

	std::map< std::string, std::map< long long, IplImage * > > queImgMap;
	std::map< IplImage *, std::pair< long long, std::string> > ptrImgMap;

	std::map< std::string, bool> registeredQueues;

	std::map< std::string, bool> registeredFloatQueues;

	 
public:


	/** Deallocates all images which are not needed by anyone. */
	void cleanup();

	static const std::string defaultSMName;

	/** If the shared memory does not exist, create the shared memory and init its basic structure. 
	If the shared memory exists, open it. 
	sizeMB - size of the shared memory in MB
	name - name of the shared memory
	destruct - if true, the shared memory is newly created even if it already exists.
	retainedHistoryMS - the size of the circular data queues in timestamp units - this is only for vector< float> queues
	*/
	TSMObject( const int sizeMB = 100, const std::string name = defaultSMName, const bool destruct = false, const int retainedHistoryMS = 5000);
		
	/** Deregisters queues and all requested images. Destroys temporal and shared memory images. All pointers to images from shared memory are invalid after this.*/
	~TSMObject();

	/*  ------------- for image queues  ------------- */ 
	
	/** Creates IplImage which resides in the mapped shared memory segment. Do not destroy this object.
		After you fill the image, you can use pushImage() to to push it into correct SM image queues.
		Or you can use discardImage() in case you decided not to use the image.
		Throws when the SM is full and the required memory space can not be allocated.
	*/	
	IplImage * createIplImage( const CvSize size, const int depth, const int channels);
	
	/** Discards image previously created by createIplImage(). The pointer is set to NULL if successfull.*/
	void discardImage( IplImage * &);
	
	/** Pushes an image previously created by createIplImage() into a SM buffer. The pointer is set to NULL if successfull.*/
	void pushImage( const std::string &queue, IplImage *&, long long timestamp);

	/** Gets the newest image in a queue. If the que is empty, the function blocks till an image is present.
	Throws when queue does not exist.*/
	void getNewest( const std::string queue, long long &timestamp, IplImage *&);

	/** Notifies that the image is no longer needed by this instance and it can be released from the memory. 
	The pointer is set to NULL and image and all references to it can not be used further. The pointer is set to NULL if successfull.*/
	void returnImage( IplImage * &img);

	/** Notifies that all images with specific timestamp are no longer needed by this instance and they can be released from the memory.
	All "IplImage *" pointers corresponding to images with this timestamp are invalid after execution (point to unallocated data).*/
	void returnTimestamp( long long ts);


	/** Tries to get an image from a specific que for specific timestamp. Three possible casses can occur:
	1) the image with requested timestamp is in the queue - returns immediately with the image
	2) the image is not yet computed, the requested timestamp corresponds to current newest timestamp - waits for the image to be computed and than returns
	3) the image is not in the queue and the timestamp is not the newest one - returns immediately with NULL pointer.
	Throws on error - queue does not exist.*/
	void getImage( const std::string queue, long long timestamp, IplImage *&);
	

	/** Marks all images with specified timestamp from regitered queues as requested by this instance. 
	These images will not be released until returnImage() or returnTimestamp() is called. 
	Also images which are not yet computed are marked as requested.
	*/
	void requestTimestamp( long long ts);	

	/** Gets the newest timestamp.  */
	long long getNewestTimestamp( );	

	/** Notify, that this instance requests data from queue with corresponding string identifier. 
	If the corresponding queue does not exist, it is created.*/
	void registerQueue( const std::string &queue);
	
	/** Notify, that this instance no longer request images with corresponding identifier.
	This does not ever destroy the queue. */
	void deregisterQueue( const std::string &queue);

	/** Get the list of currently active queues - the queues which have at least single registered consument at the moment.
		The activeQueues map contains all existing queues. Active queues have value true, passive false.
	*/
	void getActiveQueues( std::map< std::string, bool> &activeQueues);

	
	/* ------------ for float vector queues  ------------- */
	
	/** Push data into a queue.
	Throws if the queue does not exist.*/
	void push( const std::string &, const std::vector< float> &data, const long long timestamp);
	
	/**
	Tries to get data from a specific queue for specific timestamp. It newver blocks (oposite to getImage)
	Throws on error - queue does not exist.*/
	void get( const std::string &, const long long timestamp, std::vector< std::vector< float> > &);

	/**	Gets the newest data in a queue. If the queue is empty, the function blocks till the data is present.
	Throws on error - queue does not exist, this instance is not registered as receiver of the requested queue.*/
	void getNewest( const std::string &, long long &timestamp, std::vector< std::vector< float> > &);
	
	/** Gets all data with timestamps equal and newer than ... 
		Throws on error - queue does not exist, this instance is not registered as receiver of the requested queue.
	*/	
	void getNewer( const std::string &, const long long timestamp, std::multimap< long long, std::vector< float> > &);

	/** Notify, that this instance requests data from vector<float> queue with corresponding string identifier. 
	If the corresponding queue does not exist, it is created.*/
	void registerFloatQueue( const std::string &queue);
	
	/** Notify, that this instance no longer request data from vector<float> queue with corresponding identifier.
	This does not ever destroy the queue. */
	void deregisterFloatQueue( const std::string &queue);

	/** Get the list of currently active vector<float> queues - the queues which have at least single registered consument at the moment.
		The activeFloatQueues map contains all existing queues. Active queues have value true, passive false.
	*/
	void getActiveFloatQueues( std::map< std::string, bool> &activeQueues);
};


#endif


